<?php
/* --------------------------------------------------------------
   CronjobWriter.inc.php 2023-03-06
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2023 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

use Gambio\Core\Application\ValueObjects\Environment;
use Gambio\Core\Configuration\Compatibility\ConfigurationStorageRepositoryBuilder;

/**
 * Class CronjobWriter
 */
class CronjobWriter implements CronjobWriterInterface
{
    /**
     * @var \CI_DB_query_builder
     */
    protected $db;
    
    /**
     * @var ConfigurationStorageRepositoryBuilder
     */
    protected $storageBuilder;
    
    
    /**
     * CronjobWriter constructor.
     *
     * @param \CI_DB_query_builder                  $db
     * @param ConfigurationStorageRepositoryBuilder $storageBuilder
     */
    public function __construct(CI_DB_query_builder $db, ConfigurationStorageRepositoryBuilder $storageBuilder)
    {
        $this->db             = $db;
        $this->storageBuilder = $storageBuilder;
    }
    
    
    /**
     * Saves cronjob configuration into the storage.
     *
     * @param \StringType         $cronjob
     * @param \KeyValueCollection $data
     *
     * @return $this|\CronjobWriterInterface Same instance for chained method calls.
     */
    public function save(StringType $cronjob, KeyValueCollection $data)
    {
        $namespace = 'cronjobs/' . $cronjob->asString();
        $storage   = $this->storageBuilder->build($namespace);

        $data = $this->validateSettings($cronjob, $data);
        
        foreach ($data as $key => $value) {
            if ($key !== 'page_token') {
                $storage->set($key, $value);
            }
        }
        
        return $this;
    }

    public function validateSettings(StringType $cronjob, KeyValueCollection $data): KeyValueCollection
    {
        $rootDir = __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR;
        $environment = new Environment(
            file_exists($rootDir . '.dev-environment'),
            file_exists($rootDir . 'version_info/cloud.php'),
            file_exists($rootDir . '.e2e'),
        );
        if($environment->isCloud() && $cronjob->asString() === 'DeleteLogs')
        {
            if($data->getValue('olderThan') !== '2' || $data->getValue('active') !== 'true')
            {
                return new KeyValueCollection([
                    'olderThan' => '2',
                    'active'    => 'true'
                ]);
            }
        }

        if($data->keyExists('olderThan') && (int)$data->getValue('olderThan') < 1) {
            return new KeyValueCollection([
                'olderThan' => '1',
                'active'    => $data->getValue('active'),
            ]);
        }

        return $data;
    }
}